#ifndef __NUCLEO_H__
#define __NUCLEO_H__

#define FALSE false
#define TRUE true
#include "seq.h"
#include "common.h"

// Binary encoding of the nucleotides

// Generic DNA/RNA sequence
class tripletTable {
public:
  // Construct and load from a stream;
  tripletTable(istream& cin);
  
  // Generic constructor/destructor;
  tripletTable();
  virtual ~tripletTable();
  
  // read prob;
  double tripProb(int, Nucleotide, Nucleotide, Nucleotide);
    
  
  // Use for reading/writing a sequence from a stream;
  virtual void parse(istream& cin);
  

  // Operator aliases;
  friend istream& operator>>(istream& cin, tripletTable& trip);
  friend ostream& operator<<(ostream& cout, tripletTable& trip);
  
private:
  double tri1[4][4][4], tri2[4][4][4], tri3[4][4][4];
  
};

class quadTable {
public:
  // Construct and load from a stream
  quadTable(istream& cin);
 
  // Generic constructor/destructor
  quadTable();
  virtual ~quadTable();
  
  // read prob
    double quadProb(int, Nucleotide, Nucleotide, Nucleotide, Nucleotide);
    
   
  // Use for reading/writing a sequence from a stream

  virtual void parse(istream& cin);
  

  // Operator aliases
  friend istream& operator>>(istream& cin, quadTable& trip);
  friend ostream& operator<<(ostream& cout, quadTable& trip);


private:
  double quad1[4][4][4][4], quad2[4][4][4][4], quad3[4][4][4][4];
  
};

void index_to_tuple(int index, int l, int *skip, char *patt);
int tuple_to_index(int pos, int l, Sequence& seq, int *skip);
int tuple_to_index(int l, int *array, int *skip);

class Modules;

class TupleTable {
public:
  // Construct and load from a stream;
  TupleTable(istream &cin);
  
  // Generic constructor/destructor;
  TupleTable();
  ~TupleTable();
  
  // read prob;
  double tupleProbMinusOne(int frame, int index);
  double tupleProb(int frame, int index);
  double tupleProb(int frame, int pos, Sequence& seq);
  double tupleProb(int pos, Sequence& seq, int exon);

  void TupleTable::fastTupleProb(char* seq, int length, double* fprob, double* minFProb);

  double TupleTable::calcTupleRelFreq(int tupleIndex, int frame,
				      double exonWeight, double intronWeight, double penalty);
  double TupleTable::calcTupleRelFreq(Sequence& seq, int pos, int frame,
				      double exonWeigth, double intronWeight, double penalty);
  
  double TupleTable::relativeFrequency(                   Sequence& seq, int pos, int frame );
  double TupleTable::relativeFrequencyMinusOne(           Sequence& seq, int pos, int frame );
  double TupleTable::relativeFrequencyEverywhere(         Sequence& seq, int pos, int frame );
  double TupleTable::relativeFrequencyEverywhereMinusOne( Sequence& seq, int pos, int frame ); 

  double TupleTable::logFreq(Sequence& seq, int pos, int frame);

  int length() { return tuple_length; }
  void print_skips();
  void print_skips(ofstream &fout);
  
  int plusSequence(  Sequence& seq );
  int minusSequence( Sequence& seq );

  void convertToRanks();   // under development NOT TESTED !!!;
  void convertToFreqs();   // under development NOT TESTED !!!;
  
  // Operator aliases;
  friend istream& operator>>(istream &cin, TupleTable &tt);
  friend ostream& operator<<(ostream& cout, TupleTable &tt);

  int useRanks;
  int tuple_length, true_length, table_length;
  int *freqs0,*freqs1,*freqs2, *freqsI;
  int *ranks0,*ranks1,*ranks2, *ranksI;
  int *skips;
  friend class Modules;
private:
  istream& parse(istream &cin);

  int totals[4]; // totals[ 0,1,2 ] for corresponding frames, totals[3] for introns;
  int rankTotals[4];
};


class TupleEnding 
{
 public:
  TupleEnding* next;
  int freq;
  long unsigned int ending;
  int datum;

  TupleEnding(long unsigned int idx) 
    {
      ending = idx;
      freq   = 1;
      next   = NULL;
      datum  = -1;
    }

  ~TupleEnding() 
    {
      if (next) 
	{
	  delete next;
	}
      next = NULL;
    }
};

class HugeTupleEnding {

public:

  HugeTupleEnding *next;
  int freq;
  int *indices;
  int idxcnt;
  int datum;

  HugeTupleEnding() {
    indices = NULL;
    next    = NULL;
    freq=datum=idxcnt = -1;
  }
  HugeTupleEnding(int indexCount, int *idx) {
    assert(indexCount>0);
    indices = new int[indexCount];
    for (int i=0; i<indexCount; ++i) indices[i] = idx[i];
    freq   = 1;
    next   = NULL;
    datum  = -1;
    idxcnt = indexCount;
  }
  ~HugeTupleEnding() {
    
    if (next) delete next;
    next = NULL;
  }
  int hasEnding(int *idx) {
    if (idxcnt<=0)                                         return 0;
    for (int i=0; i<idxcnt; ++i) if (indices[i] != idx[i]) return 0;
    return 1;
  }
};


class LongTupleTable 
{
 public:
  TupleEnding** tupleTable;
  
  int tuple_length;
  int begin_length;
  int end_length;
  int arraylength;

  int* skips;
  
  LongTupleTable() 
    {
      tupleTable = NULL;
      tuple_length = -1;
      begin_length = -1;
      end_length = -1;
      arraylength = -1;
      skips = new int[100];
      arrayZero(skips,100);
  }

  ~LongTupleTable() 
    {
      if(tupleTable) 
	{
	  for(int i = 0; i < power(4,begin_length); i++ ) 
	    {
	      if (tupleTable[i]) delete tupleTable[i];
	    }
	  delete[] tupleTable;
	}
      if (skips) 
	{
	  delete[] skips;
	}
    }
   
  void set_lengths(int bl, int el) 
    {
      begin_length = bl;
      end_length = el;
      tuple_length = bl + el;
      arraylength = power(4,begin_length);
      tupleTable = new (TupleEnding*) [arraylength];
      for(int i = 0; i < power(4,begin_length); tupleTable[i++] = NULL);
    }

  int getDatum(int idx1, long unsigned int idx2, int &datum) 
    {
      assert(idx1 < arraylength);
      TupleEnding* te;
      if(!(te=tupleTable[idx1])) 
	{
	  return 0;
	}
      while(te) 
	{
	  if(te->ending == idx2) 
	    {
	      datum = te->datum;
	      return 1;
	    }
	  te = te->next;
	}
      return 0;
    }

  int isHit(int idx1, long unsigned int idx2) 
    {
      assert(idx1 < arraylength);
      TupleEnding* te;
      if(!(te = tupleTable[idx1])) 
	{
	  return 0;
	}
      while(te) 
	{
	  if(te->ending == idx2) 
	    {
	      return 1;
	    }
	  te = te->next;
	}
      return 0;
    }

  int isHit(int idx1, long unsigned int idx2, int cutoff) 
    {
      assert(idx1 < arraylength);
      TupleEnding* te;
      if(!(te = tupleTable[idx1])) 
	{
	  return 0;
	}
      while(te) 
	{
	  if(te->ending == idx2)
	    { 
	      if (te->freq >= cutoff) 
		{
		  return 1;
		}
	      else 
		{ 
		  return 0;
		}
	    }
	  te = te->next;
	}
    return 0;
  }

  int isHit(int pos, Sequence &seq) 
    {
      int idx1 = tuple_to_index(pos, begin_length, seq, skips);
      if(!tupleTable[idx1]) 
	{ 
	  return 0;
	}
      long unsigned int idx2 = tuple_to_index(pos+begin_length,end_length,seq,skips);
      TupleEnding* te = tupleTable[idx1];
      while(te) 
	{
	  if(te->ending == idx2) 
	    {
	      return 1;
	    }
	  te = te->next;
	}
      return 0;
    }
  
  int isHit(int pos, Sequence &seq, int cutoff) 
    {
      int idx1 = tuple_to_index(pos, begin_length, seq, skips);
      if(!tupleTable[idx1]) 
	{
	  return 0;
	}
      long unsigned int idx2 = tuple_to_index(pos+begin_length, end_length, seq, skips);
      TupleEnding* te = tupleTable[idx1];
      while(te) 
	{
	  if(te->ending == idx2 && te->freq >= cutoff) 
	    {
	      return 1;
	    }
	  te = te->next;
	}
      return 0;
    }
  
  void add_entry(int idx1, long unsigned int idx2) 
    {
      if(!tupleTable[idx1]) 
	{
	  tupleTable[idx1] = new TupleEnding(idx2);
	  return;
	}
      TupleEnding* te = tupleTable[idx1];
      while(1) 
	{
	  if (te->ending == idx2) 
	    {
	      te->freq++;
	      return;
	    }
	  if(te->next) 
	    {
	      te = te->next;
	    }
	  else 
	    {
	      te->next = new TupleEnding(idx2);
	      return;
	    }
	}
    }

  void add_entry(int idx1, long unsigned int idx2, int datum) 
    {
      if(!tupleTable[idx1]) 
	{
	  tupleTable[idx1] = new TupleEnding(idx2);
	  tupleTable[idx1]->datum = datum;
	  return;
	}
      TupleEnding* te = tupleTable[idx1];
      while(1) 
	{
	  if(te->ending == idx2) 
	    {
	      te->freq++;
	      return;
	    }
	  if(te->next) 
	    { 
	      te = te->next;
	    }
	  else 
	    {
	      te->next  = new TupleEnding(idx2);
	      te->next->datum = datum;
	      return;
	    }
	}
    }
};

istream& operator>>(istream& in,  LongTupleTable& ltt);
ostream& operator<<(ostream& out, LongTupleTable& ltt);


class HugeTupleTable {
public:
  
  HugeTupleEnding **tupleTable;
  
  int tuple_length;
  int begin_length;
  int end_length;
  int endIndicesCount;
  int arraylength;

  HugeTupleTable() {
    tupleTable = NULL;
    tuple_length=begin_length=end_length=endIndicesCount=arraylength=-1;
  }

  ~HugeTupleTable() {
    if (tupleTable) {
      for (int i = 0; i < power(4,begin_length); i++ ) if (tupleTable[i]) delete tupleTable[i];
      delete[] tupleTable;
    }
  }
   
  void set_lengths(int bl, int el, int idxcnt) {
    begin_length    = bl;
    end_length      = el;
    tuple_length    = bl + el;
    arraylength     = power(4,begin_length);
    endIndicesCount = idxcnt;
    tupleTable      = new (HugeTupleEnding*) [arraylength]; 
    for (int i=0; i<power(4,begin_length); ++i) tupleTable[i] = NULL;
  }

  int getDatum(int idx1, int *idx2, int &datum) {
    assert(idx1 < arraylength);
    HugeTupleEnding *te;
    if (!(te=tupleTable[idx1])) return 0;
    while (te) {
      if (te->hasEnding(idx2)) {
	datum = te->datum;
	return 1;
      }
      te = te->next;
    }
    return 0;
  }

  int isHit(int idx1, int *idx2) {
    assert(idx1 < arraylength);
    HugeTupleEnding *te;
    if (!(te=tupleTable[idx1])) return 0;
    while (te) {
      if (te->hasEnding(idx2)) return 1;
      te = te->next;
    }
    return 0;
  }

  int isHit(int idx1, int *idx2, int cutoff) {
    assert(idx1 < arraylength);
    HugeTupleEnding *te;
    if (!(te=tupleTable[idx1])) return 0;
    while (te) {
      if (te->hasEnding(idx2)) 
	if (te->freq >= cutoff) return 1;
	else                    return 0;
      te = te->next;
    }
    return 0;
  }

  void add_entry(int idx1, int *idx2) {
    if (!tupleTable[idx1]) {
      tupleTable[idx1] = new HugeTupleEnding(endIndicesCount, idx2);
      return;
    }
    HugeTupleEnding *te = tupleTable[idx1];
    while (1) {
      if (te->hasEnding(idx2)) {
	te->freq++;
	return;
      }
      if (te->next) te = te->next;
      else {
	te->next = new HugeTupleEnding(endIndicesCount, idx2);
	return;
      }
    }
  }
  void add_entry(int idx1, int *idx2, int datum) {
    if (!tupleTable[idx1]) {
      tupleTable[idx1] = new HugeTupleEnding(endIndicesCount, idx2);
      tupleTable[idx1]->datum = datum;
      return;
    }
    HugeTupleEnding *te = tupleTable[idx1];
    while (1) {
      if (te->hasEnding(idx2)) {
	te->freq++;
	return;
      }
      if (te->next) te = te->next;
      else {
	te->next  = new HugeTupleEnding(endIndicesCount, idx2);
	te->next->datum = datum;
	return;
      }
    }
  }
};




#endif // __NUCLEO_H__

// End of header file



