#include "glassexe.h"

void printUsage(ostream& out) 
{
  out << "\n\n";
  out << "Usage:    glass <hfile> <mfile> [options]\n\n";
  out << "Options:\n\n";
  out << "\t-o <filename>\n";
  out << "\t\tPrint output to <filename> instead of STDOUT\n\n";
  out << "\t-c\n";
  out << "\t\tPrint Clustal W header in output\n";
  out << endl;
}


void parseFromCmdLine(int argc, char* argv[], GlassExeRegistry& registry) 
{
  // Parse the optional arguments.
  int c = 0;
  while((c = getopt(argc, argv, "o:c")) != -1) {
    switch(c) {
    case 'o':
      registry.outputFileName = optarg;
      break;
    case 'c':
      registry.useClustalHeader = true;
      break;
    case '?':
      cerr << "Unknown option -" << optopt << "." << endl;
      exit(EXIT_FAILURE);
    default:
      abort();
    }
  }
  // Parse the required arguments.
  if(optind + 2 == argc) {
    registry.humanFileName = string(argv[optind]);
    registry.mouseFileName = string(argv[optind+1]);
  }
  else {
    printUsage(cerr);
    exit(EXIT_FAILURE);
  }
}


void outputAlignmentClustal(ostream &out, 
			    Sequence& hseq, Sequence& mseq, 
			    PairwiseAlignment& img) 
{
  Sequence::size_type p1 = 1;
  Sequence::size_type p2 = 1;

  Sequence::size_type hseql = hseq.size();
  Sequence::size_type mseql = mseq.size();
  int namel = MAX((hseq.name()).size(),(mseq.name()).size());
  int hpad  = namel - (hseq.name()).size() + 6;
  int mpad  = namel - (mseq.name()).size() + 6;
  int apad  = namel + 6;
  int alnl  = 50;
  int alnp  = 1;

  string halign = "";
  string malign = "";
  string aligns = "";

  while(true) {
    if((p1 > hseql) || (p2 > mseql)) {
      out << halign << "\n" << malign << "\n" << aligns << "\n" << endl;
      break;
    }

    if(alnp == 1) {
      halign = hseq.name();
      for(int i = 0; i < hpad; i++) {
	halign.append(" ");
      }
      malign = mseq.name();
      for(int i = 0; i < mpad; i++) {
	malign.append(" ");
      }
      aligns = "";
      for(int i = 0; i < apad; i++) {
	aligns.append(" ");
      }
    } 
    else if(alnp == alnl) {
      out << halign << "\n" << malign << "\n" << aligns << "\n" << endl;
      alnp = 0;
    }
    else if((img.firstImageGet(p1) > 0) && (img.secondImageGet(p2) > 0))  {
      //      assert(img.firstImageGet(p1) == p2 && img.secondImageGet(p2) == p1);
      
      halign.append(1,nucl2char(hseq[p1]));
      malign.append(1,nucl2char(mseq[p2]));
      if((hseq[p1] == mseq[p2]) && ((hseq[p1]) != BASE_UNKNOWN)) { 
	aligns.append("*");
      }
      else { 
	aligns.append(" ");
      }
      p1++; p2++;
    }
    else if((img.firstImageGet(p1) == 0) && (img.secondImageGet(p2) > 0)) {
      halign.append(1,nucl2char(hseq[p1]));
      malign.append("-");
      aligns.append(" ");
      p1++;
    }
    else if((img.firstImageGet(p1) > 0) && (img.secondImageGet(p2) == 0)) {
      halign.append("-");
      malign.append(1,nucl2char(mseq[p2]));
      aligns.append(" ");
      p2++;
    }
    if((p1 <= hseql) && (p2 <= mseql)) {
      assert(img.firstImageGet(p1) >= 0 || img.secondImageGet(p2) >= 0);
    }

    ++alnp;
  }

  out.flush();
}


int main(int argc, char* argv[]) 
{
  GlassExeRegistry registry;
  parseFromCmdLine(argc, argv, registry);

  ifstream hfile(registry.humanFileName.data());
  ifstream mfile(registry.mouseFileName.data());

  Sequence hseq = readFasta(hfile);
  Sequence mseq = readFasta(mfile);  

  Sequence* hseq_orig = new Sequence(hseq);
  Sequence* mseq_orig = new Sequence(mseq);

  Glass glass;
  // ifstream regfile("/d/tarjei/wgc/rosetta/glass.registry");   (Disabled for 1.1a)
  // glass.setParamsFromFile(regfile);
  PairwiseAlignment img = glass.align(hseq, mseq);

  if(registry.outputFileName != "") {
    ofstream fout(registry.outputFileName.c_str());
    if(registry.useClustalHeader) {
      fout << "Clustal W (1.81) multiple sequence alignment\n\n\n";
    }
    else {
      fout << "GLASS (1.1a) pairwise sequence alignment\n\n\n";
    }
    outputAlignmentClustal(fout, *hseq_orig, *mseq_orig, img);
  }
  else {
    if(registry.useClustalHeader) {
      cout << "Clustal W (1.81) multiple sequence alignment\n\n\n";
    }
    else {
      cout << "GLASS (1.1a) pairwise sequence alignment\n\n\n";
    }
    outputAlignmentClustal(cout, *hseq_orig, *mseq_orig, img);
  }

  exit(EXIT_SUCCESS);
}

