#ifndef __GLASS_H__
#define __GLASS_H__

#include <cstdlib>
#include <cstdio>
#include <unistd.h>
#include <iostream>
#include <map>
#include "common.h"
#include "glassregistry.h"
#include "nucleotable.h"
#include "pairwisealignment.h"

class Glass
{
 public:

  void setParamsFromFile(ifstream& fin);
  //
  // Modifies:  this.registry, fin
  //
  // Effects:   Calls GlassRegistry::readRegistryFile(ifstream&) to
  //            set registry fields according to an input file.
  //

  PairwiseAlignment align(Sequence& hseq, Sequence& mseq);
  //
  // Modifies:  this
  //
  // Returns:   A global PairwiseAlignment of hseq and mseq.
  //

  Glass();
  //
  // Effects:   Creates a new Glass object.
  //

 protected:

  GlassRegistry registry;

 private:

  void phaseAlign(Sequence *hseq, Sequence *mseq,
		  int phaseCount, int *tupLengths, int *extnLengths, int *extnCutoffs,
		  int *humImg, int *mouseImg, 
		  char *inval_hum_pos, char *invalid_mus_pos, 
		  int repmask_cff);
  //
  // Requires:  hseq != null &&
  //            mseq != null &&
  //            phaseCount > 0 && 
  //            tupLengths[i]  > 0 for all 0 <= i < phaseCount &&
  //            extnLengths[i] > 0 for all 0 <= i < phaseCount &&
  //            extnCutoffs[i] > 0 for all 0 <= i < phaseCount &&
  //            length(humImg[])    == hseq->size() &&
  //            length(mouseImg[i]) == mseq->size() &&
  //            inval_hum_pos[i] == 0 or 1 for 0 <= i < hseq->size() &&
  //            inval_mus_pos[i] == 0 or 1 for 0 <= i < mseq->size() &&
  //            repmask_cff >= 0
  //
  // Modifies:  humImg, mouseImg
  //
  // Effects:   Computes a global alignment of hseq and mseq that is returned
  //            in humImg and mouseImg such that
  //
  //            if hseq[i] aligns with mseq[j] for some 0 <= j < mseq,size() 
  //              humImg[i] == j && mouseImg[j] == i
  //            else
  //              humImg[i] == -1
  //
  //            and equivalently for mouseImg.
  //

  int breakptFind(int *htuples, int *hidx, int *himage, int hl,
		  int *mtuples, int *midx, int *mimage, int ml, 
		  int tl,
		  int *hbreaks, int *mbreaks);
  //
  // Requires:  length(htuples) >= hl &&
  //            length(hidx)    >= hl &&
  //            length(himage)  >= hl &&
  //            length(hbreaks) >= hl &&
  //            length(mtuples) >= ml &&
  //            length(midx)    >= ml &&
  //            length(mimage)  >= ml &&
  //            length(mbreaks) >= ml &&
  //            tl > 0
  //
  // Effects:   Puts breakpoints in hbreaks and mbreaks.
  // 
  // Returns:   Number of breakpoints found.
  //
    
  int constrainedAlign(Sequence *hseq, Sequence *mseq,
		       int *hbreaks, int *mbreaks, int breakCnt, int tupleLength,
		       double match, double mismatch, double gap, 
		       int *himage, int *mimage, int extnLength, int extnCutoff);
  
  void exactMatchFind(Sequence *hseq, Sequence *mseq, int tlength,
		      int *htuples, int *hidx, int &hl,
		      int *mtuples, int *midx, int &ml);
  //
  // Finds all tlength-long exact matches between hseq and mseq
  //
  // Requires:  hseq != null &&
  //            mseq != null &&
  //            tlength > 0  &&
  //            length(htuples[]) >= hseq->size() &&
  //            length(hidx[])    >= hseq->size() &&
  //            length(mtuples[]) >= mseq->size() &&
  //            length(midx[])    >= mseq->size()
  //
  // Modifies:  htuples, hidx, mtuples, midx
  //
  // Effects:   If tlength > 12 runs exactLongMatchFind(...) on inputs.
  //            Records tuple indices of matches in htuples and mtuples
  //            and their sequence positions in hidx and midx.
  //            hl and ml are set to the number of matching tuples.
  //

  void exactLongMatchFind(Sequence *hseq, Sequence *mseq, int tlength,
			  int *htuples, int *hidx, int &hl,
			  int *mtuples, int *midx, int &ml);
  //
  // Finds all tlength-exact mathces between hseq and mseq
  //
  // Requires:  hseq != null &&
  //            mseq != null &&
  //            tlength > 0  &&
  //            length(htuples[]) >= hseq->size() &&
  //            length(hidx[])    >= hseq->size() &&
  //            length(mtuples[]) >= mseq->size() &&
  //            length(midx[])    >= mseq->size()
  //
  // Modifies:  htuples, hidx, mtuples, midx, hl, ml
  //
  // Effects:   Records tuple indices of matches in htuples and mtuples
  //            and their sequence positions in hidx and midx.
  //            hl and ml are set to the number of matching tuples.
  //
  
  double findmap(int *seq1, int *seq2, 
		 int from1, int to1, int from2, int to2,  
		 double match, double mismatch, double gap, 
		 int *imagefirst, int *imagesecond, 
		 double *col1, double *col2, double *prebest, double *suffbest);
  // 
  // Recursive alignment helper function
  //
  // Requires:  

  double findmapUseExtensions(int *seq1, int *seq2, 
			      int from1, int to1, int from2, int to2,  
			      double match, double mismatch, double gap, 
			      int *imagefirst, int *imagesecond, 
			      double *col1, double *col2, double *prebest, double *suffbest,
			      int *hseqInt, int *mseqInt, 
			      int *hptrs, int *mptrs, 
			      int hseql, int mseql, 
			      int extnLength, int tupLength);
  // Recursive alignment helper function
  //
  // Requires:
  
  void constraintsalignUseExtensions(int *first, int *second, 
				     int length1, int length2, 
				     double match, double mismatch, double gap, 
				     int *imagefirst, int *imagesecond, 
				     int *constraints1, int *constraints2, 
				     int constraintslength, int acceptNegativeAlignments,
				     int *hseqInt, int *mseqInt, 
				     int *hptrs, int *mptrs,
				     int hseql, int mseql, 
				     int extnLength, int tupLength);
  //
  // Finds a global alignment subject to a partial given map.
  //


  double memalign(int *first, int *second, 
		  int fromfirst, int tofirst, 
		  int fromsecond, int tosecond, 
		  double match, double mismatch, double gap, 
		  int *imagefirst, int *imagesecond);
  //
  // Finds a global alignment in linear space
  //
  // Requires:  0 <= fromfirst  <= tofirst  < length(first)   &&
  //            0 <= fromSecond <= tosecond < length(second)  &&
  //            length(imagefirst)  >= tofirst  - fromfirst  + 1 &&
  //            length(imagesecond) >= tosecond - fromsecond + 1
  //
  
  double gappedAlign(int *first, int *second, 
		     int fromfirst, int tofirst, 
		     int fromsecond, int tosecond, 
		     double match, double mismatch, double gapOpen, double gap,
		     int *imagefirst, int *imagesecond);
  //
  // Performs a global alignment on two sequences. M contains the scores, P contains
  // the pointers. The encoding is 0=up, 1=left, 2=upleft.
  // The input is two sequences, and positions within them to align. The answer is returned
  // in imagefirst and imagesecond (zero based).
  //
  // Requires:  fromfirst  <= tofirst &&
  //            fromsecond <= fromsecond 
  //
  
  double tupleAlignmentScore(int *htup, int *mtup, 
			     int hfrom, int hto, int mfrom, int mto,
			     double match, double mismatch, double gap, double alpha, double beta,
			     int *himg, int *mimg);
  //
  // Calculates the score of a tuple alignment between htup[hfrom..hto] to mtup[mfrom..mto]
  //
  // Requires:  

  void Glass::setUnknownNuc(Nucleotide n, Sequence& seq, char* inval_pos);
  //
  // Sets any BASE_UNKNOWN in seq to 'n'. And marks the corresponding
  // inval_pos element.
  //

  enum alignPtrs { toD0 = 1, toD1 = 2, toU0 = 3, toU1 = 4, toL0 = 5, toL1 = 6 };

  double*** MatrixM;
  int*** MatrixP;
  int len1;
  int len2;

};


#endif
