#include "glass.h"


void Glass::setParamsFromFile(ifstream& fin)
{
  registry.readRegistryFile(fin);
}

PairwiseAlignment Glass::align(Sequence& hseq, Sequence& mseq) 
{
  int mseql = mseq.size();
  int hseql = hseq.size();
     
  int *humImg   = new int[hseql+1]; arrayInit(-1,humImg,  hseql+1);
  int *mouseImg = new int[mseql+1]; arrayInit(-1,mouseImg,mseql+1);

  char *inval_hpos = new char[hseql+1]; arrayInit((char)0, inval_hpos, hseql+1);
  char *inval_mpos = new char[mseql+1]; arrayInit((char)0, inval_mpos, mseql+1);

  setUnknownNuc(BASE_T, hseq, inval_hpos); 
  setUnknownNuc(BASE_C, mseq, inval_mpos);
    
  int minl = MIN(hseql, mseql);
  int phaseCount = 8;
  int i;
  for(i = 0; i <= 6; ++i) {
    if((16000 << i) > minl) {
      break;
    }
  }
  phaseCount += i;
  assert(phaseCount <= 14);
  
  int tupLengths[14], extnLengths[14], extnCutoffs[14];
  for (i=0; i<phaseCount; ++i) {
    tupLengths[i]  = registry.tupleLengths[i+14-phaseCount];
    extnLengths[i] = registry.extensionLengths[i+14-phaseCount];
    extnCutoffs[i] = registry.extensionCutoffs[i+14-phaseCount];
   }
  
  phaseAlign(&hseq, &mseq, 
	     phaseCount, tupLengths, extnLengths, extnCutoffs, 
	     humImg, mouseImg,
	     inval_hpos, inval_mpos, 
	     1000000);

  for (i=1; i <= hseql; ++i) {
    if (humImg[i] > 0)
      assert(humImg[i] <= mseql && mouseImg[humImg[i]] == i);
  }

  return PairwiseAlignment(humImg, mouseImg, hseql, mseql);
}


Glass::Glass() 
{
  MatrixM = NULL;
  MatrixP = NULL;
  len1 = 0;
  len2 = 0;
}


// --- LEGACY CODE FROM ALIGN.CC STARTS HERE ---

void Glass::setUnknownNuc(Nucleotide n, Sequence& seq, char* inval_pos)
{
  for(Sequence::size_type i = 0; i < seq.size(); i++) {
    if(seq[i] == BASE_UNKNOWN) {
      seq[i] = n;
      inval_pos[i] = 1;
    }
  }
}

void Glass::phaseAlign(Sequence *hseq, Sequence *mseq,
		       int phaseCount, int *tupLengths, int *extnLengths, int *extnCutoffs,
		       int *humImg, int *mouseImg,
		       char *inval_hum_pos, char *inval_mus_pos, 
		       int repmask_cff) 
{
  int ph = 0;
  int hseql = hseq->size();
  int mseql = mseq->size();
  int currLength;
  int lastConstrPos;
  double alignScoreLeft;
  double alignScoreRight;

  int* hseqInt = new int[hseql+1]; seq2int(hseq, hseqInt);
  int* mseqInt = new int[mseql+1]; seq2int(mseq, mseqInt);

  int** htuples = new (int*)[phaseCount]; int** hidx = new (int*)[phaseCount];
  int** mtuples = new (int*)[phaseCount]; int** midx = new (int*)[phaseCount];

  for(int i = 0; i < phaseCount; ++i) {
    htuples[i] = new int[hseql+1]; arrayZero(htuples[i], hseql+1);
    mtuples[i] = new int[mseql+1]; arrayZero(mtuples[i], mseql+1);
    hidx[i]    = new int[hseql+1]; arrayZero(hidx[i],    hseql+1);
    midx[i]    = new int[mseql+1]; arrayZero(midx[i],    mseql+1);
  }
  int *hl = new int[phaseCount]; arrayZero(hl, phaseCount);
  int *ml = new int[phaseCount]; arrayZero(ml, phaseCount);
  
  for(int i = 0; i < phaseCount; ++i) {

    exactMatchFind(hseq, mseq, 
		   tupLengths[i], 
		   htuples[i], hidx[i], hl[i], 
		   mtuples[i], midx[i], ml[i]);

    if(tupLengths[i] >= repmask_cff) {
      bool hasrep;
      for(int j = 0; j < hl[i]; ++j) {
	hasrep = false;
	for(int k = hidx[i][j]; k < hidx[i][j] + tupLengths[i]; ++k) {
	  if(inval_hum_pos[k]) {
	    hasrep = true;
	  }
	}
	if(hasrep) {
	  htuples[i][j] = -1;
	}
      }
      for(int j = 0; j < ml[i]; ++j) {
	hasrep = false;
	for(int k = midx[i][j]; k < midx[i][j] + tupLengths[i]; ++k) {
	  if(inval_mus_pos[k]) {
	    hasrep = true;
	  }
	}
	if(hasrep) {
	  mtuples[i][j] = -1;
	}
      }
      int currptrh = 0;
      for(int j = 0; j < hl[i]; ++j) {
	if(htuples[i][j] >= 0) {
	  htuples[i][currptrh] = htuples[i][j];
	  hidx   [i][currptrh] = hidx   [i][j];
	  currptrh++;
	}
      }
      int currptrm = 0;
      for(int j = 0; j < ml[i]; ++j) {
	if (mtuples[i][j] >= 0) {
	  mtuples[i][currptrm] = mtuples[i][j];
	  midx   [i][currptrm] = midx   [i][j];
	  currptrm++;
	}
      }
      hl[i] = currptrh;
      ml[i] = currptrm;
    }
  }

  int** htupImg = new (int*)[phaseCount]; int** hbreaks = new (int*)[phaseCount];
  int** mtupImg = new (int*)[phaseCount]; int** mbreaks = new (int*)[phaseCount];
  for(int i = 0; i < phaseCount; ++i) {
    htupImg[i] = new int[ hl[i] ]; arrayInit(-1, htupImg[i], hl[i]);
    hbreaks[i] = new int[ hl[i] ]; arrayInit(-1, hbreaks[i], hl[i]);
    mtupImg[i] = new int[ ml[i] ]; arrayZero(mtupImg[i], ml[i]);
    mbreaks[i] = new int[ ml[i] ]; arrayZero(mbreaks[i], ml[i]);
  }
  int* breakCnt        = new int[phaseCount]; arrayZero(breakCnt, phaseCount);
  int* hconstrExplicit = new int[hseql+1];
  int* hinverseIdx     = new int[hseql+1];
  int* minverseIdx     = new int[mseql+1];
  int* hconstraints    = new int[hseql+1];
  int* mconstraints    = new int[mseql+1];
  int* himgFixed       = new int[hseql+1]; arrayInit(-2, himgFixed, hseql+1);
  int* mimgFixed       = new int[mseql+1]; arrayInit(-2, mimgFixed, mseql+1);

  bool firstPass = false;
  double firstPassScore = 0;
        
  for(ph = 0; ph < phaseCount; ++ph) {
    if (!firstPass) {
      firstPassScore = memalign(htuples[0], mtuples[0], 
				0, hl[0]-1, 0, ml[0]-1, 
				registry.matchScore, registry.mismatchScore, registry.gapScore, 
				htupImg[0], mtupImg[0]);

      firstPassScore = tupleAlignmentScore(htuples[0], mtuples[0], 
					   0, hl[0]-1, 0, ml[0]-1, 
					   registry.matchScore, registry.mismatchScore, 
					   registry.gapScore, registry.alpha, registry.beta, 
					   htupImg[0], mtupImg[0]);
      firstPass = true;
      continue;
    }

    int phtl   = tupLengths[ph-1];
    currLength = extnLengths[ph-1];

    breakCnt[ph-1] = breakptFind(htuples[ph-1], hidx[ph-1], htupImg[ph-1], hl[ph-1],
				 mtuples[ph-1], midx[ph-1], mtupImg[ph-1], ml[ph-1], 
				 phtl, hbreaks[ph-1], mbreaks[ph-1]);

    int* breaksValid = new int[breakCnt[ph-1]]; arrayZero(breaksValid, breakCnt[ph-1]);

    for(int i = 0; i < breakCnt[ph-1]; ++i) {

      if((himgFixed[hbreaks[ph-1][i]] == mbreaks[ph-1][i]) || 
	 (himgFixed[hbreaks[ph-1][i] + phtl - 1] == mbreaks[ph-1][i] + phtl - 1)) {
	breaksValid[i] = 1;
	continue;
      }

      if ((hbreaks[ph-1][i] <= extnLengths[ph-1]) || 
	  (hbreaks[ph-1][i] + phtl >= hseql-extnLengths[ph-1]) ||
	  (mbreaks[ph-1][i] <= extnLengths[ph-1]) || 
	  (mbreaks[ph-1][i] + phtl >= mseql-extnLengths[ph-1])) {
	breaksValid[i] = 1;
	continue;
      }

      int* himageLeft = new int[currLength]; int* himageRight = new int[currLength];
      int* mimageLeft = new int[currLength]; int* mimageRight = new int[currLength];
      int disagreeWithFixedLeft  = 0;
      int disagreeWithFixedRight = 0;
      
      arrayZero(himageLeft,  currLength); arrayZero(himageRight, currLength);
      arrayZero(mimageLeft,  currLength); arrayZero(mimageRight, currLength);
      	
      alignScoreLeft  = gappedAlign(hseqInt, mseqInt,
				    hbreaks[ph-1][i] - currLength, hbreaks[ph-1][i] - 1,
				    mbreaks[ph-1][i] - currLength, mbreaks[ph-1][i] - 1,
				    registry.matchScore, registry.mismatchScore, 
				    registry.gapOpenScore, registry.gapScore,
				    himageLeft, mimageLeft);

      for(int j = currLength; j > 0; --j) {
	if ((himgFixed[hbreaks[ph-1][i] - j] > 0) &&
	    ((himageLeft[currLength - j] == -1) || 
	     (himgFixed[hbreaks[ph-1][i] - j] != 
	      mbreaks[ph-1][i] - currLength + himageLeft[currLength - j]))) {
	  disagreeWithFixedLeft = 1;
	}
      }

      alignScoreRight = gappedAlign(hseqInt, mseqInt,
				    hbreaks[ph-1][i] + phtl, hbreaks[ph-1][i] + phtl + currLength-1,
				    mbreaks[ph-1][i] + phtl, mbreaks[ph-1][i] + phtl + currLength-1,
				    registry.matchScore, registry.mismatchScore, 
				    registry.gapOpenScore, registry.gapScore,
				    himageRight, mimageRight);

      for(int j = 0; j < currLength; ++j) {
	if((himgFixed[hbreaks[ph-1][i] + phtl + j] > 0) &&
	   ((himageRight[j] == -1) || 
	    (himgFixed[hbreaks[ph-1][i] + phtl + j] != 
	     mbreaks[ph-1][i] + phtl + himageLeft[j]))) {
	  disagreeWithFixedRight = 1;
	}
      }

      if(alignScoreLeft + alignScoreRight >= extnCutoffs[ph-1] && 
	 !disagreeWithFixedLeft && !disagreeWithFixedRight) {
	breaksValid[i] = 1;
      }
      delete[] himageLeft; delete[] himageRight;
      delete[] mimageLeft; delete[] mimageRight;
    }

    arrayInit(-1,hconstrExplicit, hseql+1);

    for(int i = 0; i < breakCnt[ph-1]; ++i) {
      if(breaksValid[i]) {
	for(int j = hbreaks[ph-1][i]; j < hbreaks[ph-1][i] + phtl; ++j) {
	  himgFixed[j] = mbreaks[ph-1][i] + (j-hbreaks[ph-1][i]);
	  mimgFixed[mbreaks[ph-1][i] + (j-hbreaks[ph-1][i])] = j;
	}
	for(int j = 0; j <= tupLengths[ph-1]-tupLengths[ph]; ++j) {
	  hconstrExplicit[hbreaks[ph-1][i]+j] = mbreaks[ph-1][i]+j;
	}
      }
    }
    delete[] breaksValid;

    lastConstrPos = -10000;
    for(int i = 2; i < hseql+1; ++i) {
      if(lastConstrPos > i - tupLengths[ph]) {
	if((hconstrExplicit[i]   == -1) && 
	   (hconstrExplicit[i+1] >= 0)  &&
	   (i+1 - lastConstrPos == hconstrExplicit[i+1] - hconstrExplicit[lastConstrPos])) {
	  for(int j = 1; j + lastConstrPos <= i; ++j) {
	    hconstrExplicit[j+lastConstrPos] = hconstrExplicit[lastConstrPos] + j;
	  }
	}
      }
      if(hconstrExplicit[i] >= 0) {
	lastConstrPos = i;
      }
    }

    arrayInit(-1, hinverseIdx, hseql+1); 
    for(int i = 0; i < hl[ph]; ++i) {
      hinverseIdx[hidx[ph][i]] = i;
    }
    arrayInit(-1, minverseIdx, mseql+1); 
    for(int i = 0; i < ml[ph]; ++i) {
      minverseIdx[midx[ph][i]] = i;
    }

    arrayZero(hconstraints, hseql+1);
    arrayZero(mconstraints, mseql+1);
    
    int constraintCnt = 0;
    
    for(int i = 1; i <= hseql; ++i) {
      if(hconstrExplicit[i] >= 0) {
	hconstraints[constraintCnt] = hinverseIdx[i];
	mconstraints[constraintCnt] = minverseIdx[hconstrExplicit[i]];
	constraintCnt++;
      }
    }

   constraintsalignUseExtensions(htuples[ph], mtuples[ph], hl[ph], ml[ph], 1, 0, 0, 
				 htupImg[ph], mtupImg[ph],
				 hconstraints, mconstraints, constraintCnt, 1,
				 hseqInt, mseqInt, hidx[ph], midx[ph], hseql, mseql, 
				 extnLengths[ph], tupLengths[ph]);
  }  

  ph = phaseCount - 1;
  breakCnt[ph] = breakptFind(htuples[ph], hidx[ph], htupImg[ph], hl[ph],
			     mtuples[ph], midx[ph], mtupImg[ph], ml[ph], 
			     tupLengths[ph], hbreaks[ph], mbreaks[ph]);

  int phtl = tupLengths[ph];
  int* breaksValid = new int[breakCnt[ph]]; arrayZero(breaksValid, breakCnt[ph]);

  for(int i = 0; i < breakCnt[ph]; ++i) {
    if(hbreaks[ph][i] <= extnLengths[ph] || hbreaks[ph][i] + phtl >= hseql-extnLengths[ph] ||
       mbreaks[ph][i] <= extnLengths[ph] || mbreaks[ph][i] + phtl >= mseql-extnLengths[ph]) {
      breaksValid[i] = 1;
      continue;
    }

    currLength = extnLengths[ph];
    int* himageLeft = new int[currLength]; int* himageRight = new int[currLength];
    int* mimageLeft = new int[currLength]; int* mimageRight = new int[currLength];
    int disagreeWithFixedLeft  = 0;
    int disagreeWithFixedRight = 0;

    arrayZero(himageLeft,  currLength); arrayZero(himageRight, currLength);
    arrayZero(mimageLeft,  currLength); arrayZero(mimageRight, currLength);
    
    alignScoreLeft  = gappedAlign(hseqInt, mseqInt,
				  hbreaks[ph][i] - currLength, hbreaks[ph][i] - 1,
				  mbreaks[ph][i] - currLength, mbreaks[ph][i] - 1,
				  registry.matchScore, registry.mismatchScore, 
				  registry.gapOpenScore, registry.gapScore,
				  himageLeft, mimageLeft);
    
    for(int j = currLength; j > 0; --j) {
      if (himgFixed[hbreaks[ph][i] - j] > 0 &&
	  (himageLeft[currLength - j] == -1 || 
	   himgFixed[hbreaks[ph][i] - j] != 
	   mbreaks[ph][i] - currLength + himageLeft[currLength - j])) {
	disagreeWithFixedLeft = 1;
      }
    }

    alignScoreRight = gappedAlign(hseqInt, mseqInt,
				  hbreaks[ph][i] + phtl, hbreaks[ph][i] + phtl + currLength-1,
				  mbreaks[ph][i] + phtl, mbreaks[ph][i] + phtl + currLength-1,
				  registry.matchScore, registry.mismatchScore, 
				  registry.gapOpenScore, registry.gapScore,
				  himageRight, mimageRight);
    
    for(int j = 0; j < currLength; ++j) {
      if((himgFixed[hbreaks[ph][i] + phtl + j] > 0) &&
	 ((himageRight[j] == -1) || 
	  (himgFixed[hbreaks[ph][i] + phtl + j] != 
	   mbreaks[ph][i] + phtl + himageLeft[j]))) {
	disagreeWithFixedRight = 1;
      }
    }

    if((alignScoreLeft + alignScoreRight >= extnCutoffs[ph]) && 
       !disagreeWithFixedLeft && 
       !disagreeWithFixedRight) {
      breaksValid[i] = 1;
    }
    delete[] himageLeft; delete[] himageRight;
    delete[] mimageLeft; delete[] mimageRight;
  }

  arrayInit(-1,hconstrExplicit, hseql+1);

  for(int i = 0; i < breakCnt[ph]; ++i) {
    if (breaksValid[i]) {
      for(int j = 0; j <= tupLengths[ph]-1; ++j) {
	hconstrExplicit[hbreaks[ph][i]+j] = mbreaks[ph][i]+j;
      }
    }
  }
  delete[] breaksValid;
  
  arrayZero(hconstraints, hseql+1);
  arrayZero(mconstraints, mseql+1);
  
  int constraintCnt = 0;
  
  for(int i = 1; i <= hseql; ++i) {
    if(hconstrExplicit[i] >= 0) {
      hconstraints[constraintCnt] = i;                   humImg[i]           = hconstrExplicit[i];
      mconstraints[constraintCnt] = hconstrExplicit[i];  mouseImg[humImg[i]] = i;
      constraintCnt++;
    }
  }

  constrainedAlign(hseq, mseq, hconstraints, mconstraints, constraintCnt, 
		   1, 1, -1, -2, humImg, mouseImg, 7, 0);
  
  for(int i = 0; i < phaseCount; ++i) {
    delete[] htuples[i]; delete[] hidx[i]; delete[] htupImg[i]; delete[] hbreaks[i];
    delete[] mtuples[i]; delete[] midx[i]; delete[] mtupImg[i]; delete[] mbreaks[i];
  }
  delete[] htuples; delete[] hidx; delete[] htupImg; delete[] hbreaks;
  delete[] mtuples; delete[] midx; delete[] mtupImg; delete[] mbreaks;
  
  delete[] hl; delete[] ml;
  delete[] himgFixed;
  delete[] mimgFixed;
  
  delete[] breakCnt;
  delete[] hconstrExplicit;
  delete[] hinverseIdx;
  delete[] minverseIdx;
  delete[] hconstraints;
  delete[] mconstraints;
}



int Glass::breakptFind(int* htuples, int* hidx, int* himage, int hl,
		       int* mtuples, int* midx, int* mimage, int ml, 
		       int tl,
		       int* hbreaks, int* mbreaks) 
{  
  // Find matches;
  int* hmatched = new int[hl]; arrayZero(hmatched,hl);
  int* mmatched = new int[ml]; arrayZero(mmatched,ml);

  for(int i = 0; i < hl; ++i) {
    if(himage[i] >= 0) {
      if(htuples[i] == mtuples[himage[i]]) {
	hmatched[i] = mmatched[himage[i]] = 1;
      }
    }
  }

  // Find consistent matches;
  int* hconsistent = new int[hl]; arrayInit(1, hconsistent, hl);
  int* mconsistent = new int[ml]; arrayInit(1, mconsistent, ml);

  for(int i = 0; i < hl; ++i) { if(!hmatched[i]) { hconsistent[i] = 0; } }
  for(int i = 0; i < ml; ++i) { if(!mmatched[i]) { mconsistent[i] = 0; } }

  for(int i = 1; i < hl; ++i) {
    if(hmatched[i]) {
      int j = i - 1; 
      while(j >= 0) {
	if(hmatched[j]) {
	  if(hidx[i] - hidx[j] >= tl) { break; }
	  if(midx[himage[i]] - midx[himage[j]] != hidx[i] - hidx[j]) {
	    hconsistent[i] = hconsistent[j] = 0;
	    mconsistent[himage[i]] = mconsistent[himage[j]] = 0;
	  }
	}
	j--;
      }
    }
  }
  for(int i = 1; i < ml; ++i) { 
    if(mmatched[i]) {
      int j = i - 1; 
      while(j >= 0) {
	if(mmatched[j]) {
	  if(midx[i] - midx[j] >= tl) break;
	  if(midx[i] - midx[j] != hidx[mimage[i]] - hidx[mimage[j]]) {
	    mconsistent[i] = mconsistent[j] = 0;
	    hconsistent[mimage[i]] = hconsistent[mimage[j]] = 0;
	  }
	}
	j--;
      }
    }
  }

  // Save results to hbreaks, mbreaks;  
  int breakCnt = 0;  
  for(int i = 0; i < hl; ++i) {
    if(hconsistent[i]) {
      hbreaks[breakCnt] = hidx[i];
      mbreaks[breakCnt] = midx[himage[i]];  
      breakCnt++;
    }
  }
  
  delete[] hmatched; delete[] hconsistent;
  delete[] mmatched; delete[] mconsistent;
  
  return breakCnt;
}


int Glass::constrainedAlign(Sequence *hseq, Sequence *mseq,
			    int *hbreaks, int *mbreaks, 
			    int breakCnt, int tuplength,
			    double match, double mismatch, double gap, 
			    int *himage, int *mimage, 
			    int extnLength = 0, int extnCutoff = -10000) {
  int hseql = hseq->size();
  int mseql = mseq->size();
  
  arrayInit(-1,himage,hseql+1);
  arrayInit(-1,mimage,mseql+1);

  int* hseqint = new int[hseql+1]; seq2int(hseq,hseqint);
  int* mseqint = new int[mseql+1]; seq2int(mseq,mseqint);
  double localScore = 0;

  // First alignment;
  
  int extended = 0;
  if((hbreaks[0] > 1) && (mbreaks[0] > 1)) {
    // extensions BEGIN;
    if(extnLength > 0 && 
       hbreaks[0] > extnLength && 
       mbreaks[0] > extnLength) {
      localScore = gappedAlign(hseqint, mseqint, 
			       hbreaks[0]-extnLength, hbreaks[0]-1,
			       mbreaks[0]-extnLength, mbreaks[0]-1,
			       match, mismatch, 3*gap, gap, 
			       himage+hbreaks[0]-extnLength, mimage+mbreaks[0]-extnLength);
      if(localScore >= extnCutoff) {
	extended = 1;
	for(int j = hbreaks[0]-extnLength; j < hbreaks[0]; ++j) {
	  if(himage[j] >= 0) {
	    himage[j] += mbreaks[0]-extnLength;
	  }
	}
	for(int j = mbreaks[0]-extnLength; j < mbreaks[0]; ++j) {
	  if(mimage[j] >= 0) {
	    mimage[j] += hbreaks[0]-extnLength;
	  }
	}

	if((hbreaks[0] - extnLength > 1) && (mbreaks[0] - extnLength > 1)) {
	  gappedAlign(hseqint, mseqint, 
		      1, hbreaks[0]-extnLength-1, 
		      1, mbreaks[0]-extnLength-1, 
		      match, mismatch, 2*gap, gap, 
		      himage+1, mimage+1);
	  for(int j = 1; j < hbreaks[0]-extnLength; ++j) { if(himage[j] >= 0) { himage[j]++; } }
	  for(int j = 1; j < mbreaks[0]-extnLength; ++j) { if(mimage[j] >= 0) { mimage[j]++; } }
	}
      }
    }
    if(!extended) {
      for(int j = 1; j < hbreaks[0]; himage[j++] = -1);
      for(int j = 1; j < mbreaks[0]; mimage[j++] = -1);
      
      gappedAlign(hseqint, mseqint, 
		  1, hbreaks[0]-1, 
		  1, mbreaks[0]-1, 
		  match, mismatch, 2*gap, gap, 
		  himage+1, mimage+1);
      for(int j = 1; j < hbreaks[0]; ++j) { if(himage[j] >= 0) { himage[j]++; } }
      for(int j = 1; j < mbreaks[0]; ++j) { if(mimage[j] >= 0) { mimage[j]++; } }
    }
  }
  
  // Middle alignments;
  int accepted1 = 0;
  int accepted2 = 0;
  int from1, from2, to1, to2;
  for(int i = 0; i < breakCnt - 1; ++i) {
    for(int j = 0; j < tuplength; ++j) {
      himage[hbreaks[i] + j] = mbreaks[i] + j;
      mimage[mbreaks[i] + j] = hbreaks[i] + j;
    }
    if((hbreaks[i] + tuplength < hbreaks[i+1]) && 
       (mbreaks[i] + tuplength < mbreaks[i+1])) {
      accepted1 = 0; 
      accepted2 = 0;
      if((extnLength > 0) && 
	 (hbreaks[i] + tuplength + 2*extnLength < hbreaks[i+1]) &&
	 (mbreaks[i] + tuplength + 2*extnLength < mbreaks[i+1])) {
	localScore = gappedAlign(hseqint, mseqint, 
				 hbreaks[i] + tuplength, hbreaks[i]+tuplength+extnLength-1,
				 mbreaks[i] + tuplength, mbreaks[i]+tuplength+extnLength-1,
				 match, mismatch, 3*gap, gap, 
				 himage + hbreaks[i] + tuplength, mimage + mbreaks[i] + tuplength);
	if(localScore > extnCutoff) {
	  accepted1 = 1;
	}
	localScore = gappedAlign(hseqint, mseqint, 
				 hbreaks[i+1]-extnLength, hbreaks[i+1]-1,
				 mbreaks[i+1]-extnLength, mbreaks[i+1]-1,
				 match, mismatch, 3*gap, gap, 
				 himage + hbreaks[i+1]-extnLength, mimage + mbreaks[i+1]-extnLength);
	if(localScore > extnCutoff) {
	  accepted2 = 1;
	}
      }
      if(accepted1) {
	from1 = hbreaks[i]+tuplength+extnLength;
	from2 = mbreaks[i]+tuplength+extnLength;
      }
      else {
	from1 = hbreaks[i] + tuplength;
	from2 = mbreaks[i] + tuplength;
      }
      if (accepted2) {
	to1 = hbreaks[i+1]-extnLength-1;
	to2 = mbreaks[i+1]-extnLength-1;
      }
      else {
	to1 = hbreaks[i+1]-1;
	to2 = mbreaks[i+1]-1;
      }
      for(int j = hbreaks[i]+tuplength; j < from1; ++j) { 
	if(himage[j] >= 0) 
	  { 
	    himage[j] += mbreaks[i] + tuplength;
	  }
      }
      for(int j = mbreaks[i]+tuplength; j < from2; ++j) {
	if(mimage[j] >= 0) {
	  mimage[j] += hbreaks[i] + tuplength;
	}
      }

      for(int j = from1; j <= to1; ++j) { himage[j] = -1; }
      for(int j = from2; j <= to2; ++j) { mimage[j] = -1; }
      
      for(int j = to1+1; j < hbreaks[i+1]; ++j) { if(himage[j] >= 0) { himage[j] += to2+1; } }
      for(int j = to2+1; j < mbreaks[i+1]; ++j) { if(mimage[j] >= 0) { mimage[j] += to1+1; } }

      if((from1 <= to1) && (from2 <= to2)) {
	gappedAlign(hseqint, mseqint, 
		    from1, to1, from2, to2,
		    match, mismatch, 2*gap, gap, 
		    himage + from1, mimage + from2);
      }
      for(int j = from1; j <= to1; ++j) { if(himage[j] >= 0) { himage[j] += from2; } }
      for(int j = from2; j <= to2; ++j) { if(mimage[j] >= 0) { mimage[j] += from1; } }
    }
  }

  int i = breakCnt - 1;

  // Last alignment;
  for(int j = 0; j < tuplength; ++j) {
    himage[hbreaks[i]+j] = mbreaks[i]+j;
    mimage[mbreaks[i]+j] = hbreaks[i]+j;
  }
  
  if((hbreaks[i] + tuplength <= hseql) && (mbreaks[i] + tuplength <= mseql)) {
    extended = 0;
    if((extnLength > 0) && 
       (hbreaks[i] + tuplength + extnLength-1 <= hseql) &&
       (mbreaks[i] + tuplength + extnLength-1 <= mseql)) {
      localScore = gappedAlign(hseqint, mseqint, 
			       hbreaks[i]+tuplength, hbreaks[i]+tuplength+extnLength-1,
			       mbreaks[i]+tuplength, mbreaks[i]+tuplength+extnLength-1,
			       match, mismatch, 3*gap, gap, 
			       himage+hbreaks[i]+tuplength, mimage+mbreaks[i]+tuplength);
      if(localScore >= extnCutoff) {
	extended = 1;
	for(int j = hbreaks[i] + tuplength; j < hbreaks[i] + tuplength + extnLength; ++j) {
	  if(himage[j] >= 0) {
	    himage[j] += mbreaks[i] + tuplength;
	  }
	}
	for(int j = mbreaks[i] + tuplength; j < mbreaks[i] + tuplength + extnLength; ++j) {
	  if(mimage[j] >= 0) {
	    mimage[j] += hbreaks[i] + tuplength;
	  }
	}

	if((hbreaks[i]+tuplength+extnLength <= hseql) && 
	   (mbreaks[i]+tuplength+extnLength <= mseql)) {
	  gappedAlign(hseqint, mseqint, 
		      hbreaks[i]+tuplength+extnLength, hseql, 
		      mbreaks[i]+tuplength+extnLength, mseql,
		      match, mismatch, 2*gap, gap, 
		      himage+hbreaks[i]+tuplength+extnLength, mimage+mbreaks[i]+tuplength+extnLength);
	}
	for(int j = hbreaks[i] + tuplength + extnLength; j <= hseql; ++j) {
	  if(himage[j] >= 0) {
	    himage[j] += mbreaks[i] + tuplength + extnLength;
	  }
	}
	for(int j = mbreaks[i] + tuplength + extnLength; j <= mseql; ++j) {
	  if(mimage[j] >= 0) {
	    mimage[j] += hbreaks[i] + tuplength + extnLength;
	  }
	}
      }
    }

    if(!extended) {
      for(int j = hbreaks[i]+tuplength; j<=hseql; ++j) { himage[j] = -1; }
      for(int j = mbreaks[i]+tuplength; j<=mseql; ++j) { mimage[j] = -1; }

      gappedAlign(hseqint, mseqint, 
		  hbreaks[i]+tuplength, hseql, 
		  mbreaks[i]+tuplength, mseql,
		  match, mismatch, 2*gap, gap, 
		  himage+ hbreaks[i] + tuplength, mimage + mbreaks[i] + tuplength);
      for(int j = hbreaks[i] + tuplength; j <= hseql; ++j) {
	if(himage[j] >= 0) {
	  himage[j] += mbreaks[i] + tuplength;
	}
      }
      for(int j = mbreaks[i] + tuplength; j <= mseql; ++j) {
	if(mimage[j] >= 0) {
	  mimage[j] += hbreaks[i] + tuplength;
	}
      }
    }
  }
  
  delete[] hseqint;
  delete[] mseqint;

  return 1;
}



void Glass::exactMatchFind(Sequence *hseq, Sequence *mseq, int tlength,
			  int *htuples, int *hidx, int &hl,
			  int *mtuples, int *midx, int &ml) 
{

  if(tlength > 12) {
    return exactLongMatchFind(hseq, mseq, tlength, htuples, hidx, hl, mtuples, midx, ml);
  }

  int hseqlen = hseq->size();
  int mseqlen = mseq->size();

  int* htuplesAll = new int[hseqlen+1]; arrayZero(htuplesAll,hseqlen+1);      
  int* mtuplesAll = new int[mseqlen+1]; arrayZero(mtuplesAll,mseqlen+1);
  
  int ttl = power(4,tlength);
  int* tt = new int[ttl]; arrayZero(tt,ttl);
  
  for(int i = 1; i <= hseqlen-tlength; ++i) htuplesAll[i] = patt_to_index(*hseq,i,tlength);
  for(int i = 1; i <= mseqlen-tlength; ++i) mtuplesAll[i] = patt_to_index(*mseq,i,tlength);
  for(int i = 1; i <= hseqlen-tlength; ++i) tt[htuplesAll[i]] = 1;

  hl = ml = 0;
  
  for(int i = 1; i <= mseqlen - tlength; ++i) {
    if(mtuplesAll[i] >= 0 && (mtuplesAll[i] < ttl) && tt[mtuplesAll[i]]) { 
      tt[mtuplesAll[i]] = 2; 
      ml++; 
    }
    else {
      mtuplesAll[i] = -1;
    }
  }

  for(int i = 1; i < hseqlen - tlength + 1; ++i) {
    if(htuplesAll[i] >= 0 && (htuplesAll[i] < ttl) && tt[htuplesAll[i]] == 2) {
      hl++;
    }
    else {
      htuplesAll[i] = -1;  
    }
  }

  int hptr = 0;
  int mptr = 0;

  for(int i = 1; i <= hseqlen - tlength; ++i) {
    if (htuplesAll[i] >= 0) {
      htuples[hptr] = htuplesAll[i];
      hidx[hptr]    = i;
      hptr++;
    }
  }

  for(int i = 1; i <= mseqlen - tlength; ++i) {
    if (mtuplesAll[i]>=0) {
      mtuples[mptr] = mtuplesAll[i];
      midx[mptr]    = i;
      mptr++;
    }
  }

  delete[] htuplesAll;
  delete[] mtuplesAll;
  delete[] tt;
}

void Glass::exactLongMatchFind(Sequence *hseq, Sequence *mseq, int tlength,
			      int *htuples, int *hidx, int &hl,
			      int *mtuples, int *midx, int &ml) 
{
  int hseqlen = hseq->size();
  int mseqlen = mseq->size();

  HugeTupleTable *ltt = new HugeTupleTable();

  int tupIdxcnt = (tlength-11)/12 + ( (tlength-11)%12 ? 1 : 0 );
  int lastIdxl  = (tlength-11)%12 ? (tlength-11)%12 : 12;

  ltt->set_lengths(11, 12, tupIdxcnt);

  int** htuplesAll = new (int*)[tupIdxcnt+1];
  int** mtuplesAll = new (int*)[tupIdxcnt+1];
  
  for(int i = 0; i <= tupIdxcnt; ++i) {
    htuplesAll[i] = new int[hseqlen+1]; arrayZero(htuplesAll[i], hseqlen+1);
    mtuplesAll[i] = new int[mseqlen+1]; arrayZero(mtuplesAll[i], mseqlen+1);
  }  

  for(int i = 1; i <= hseqlen-tlength; ++i) {
    htuplesAll[0][i] = (int)patt_to_index(*hseq, i,    11);
    for(int j = 0; j < tupIdxcnt-1; ++j) {
      htuplesAll[ j+1 ][i] = patt_to_index(*hseq, i+11 + j*12, 12);
    }
    htuplesAll[ tupIdxcnt ][i]   = patt_to_index(*hseq, i+11 + (tupIdxcnt - 1)*12, lastIdxl);
  }

  for(int i = 1; i <= mseqlen-tlength; ++i) {
    mtuplesAll[0][i] = patt_to_index(*mseq, i, 11);
    for(int j = 0; j < tupIdxcnt - 1; ++j) {
      mtuplesAll[ j+1 ][i] = patt_to_index(*mseq, i+11 + j*12, 12);
    }
    mtuplesAll[ tupIdxcnt ][i]   = patt_to_index(*mseq, i+11 + (tupIdxcnt - 1)*12, lastIdxl);
  }

  for(int i = 1; i <= hseqlen - tlength; ++i) {
    int indices[100]; for(int j = 0; j < tupIdxcnt; ++j) { indices[j] = htuplesAll[j+1][i]; }
    if(!ltt->isHit(htuplesAll[0][i], indices)) {
      ltt->add_entry(htuplesAll[0][i], indices, i);
    }
  }
  hl = ml = 0;
  
  for(int i = 1; i <= mseqlen-tlength; ++i) {
    if(mtuplesAll[0][i] >= 0) {
      int indices[100]; for(int j = 0; j < tupIdxcnt; ++j) { indices[j] = mtuplesAll[j+1][i]; }
      if(ltt->isHit(mtuplesAll[0][i], indices)) {
	ltt->add_entry(mtuplesAll[0][i], indices);
	ml++;
      }
      else { 
	mtuplesAll[0][i] = -1;
      }
    }
    else {
      mtuplesAll[0][i] = -1;
    }
  }

  for(int i = 1; i < hseqlen - tlength + 1; ++i) {
    if(htuplesAll[0][i] >= 0) {
      int indices[100]; for(int j = 0; j < tupIdxcnt; ++j) { indices[j] = htuplesAll[j+1][i]; }
      if(ltt->isHit(htuplesAll[0][i], indices, 2)) {
	hl++;
      }
      else {
	htuplesAll[0][i] = -1;
      }
    }
    else {
      htuplesAll[0][i] = -1;
    }
  }

  int hptr = 0;
  int mptr = 0;
  for(int i = 1; i <= hseqlen-tlength; ++i) {
    if(htuplesAll[0][i] >= 0) {
      int indices[100]; for(int j = 0; j < tupIdxcnt; ++j) { indices[j] = htuplesAll[j+1][i]; }
      hidx[hptr] = i;
      hptr++;
    }
  }

  for(int i = 1; i <= mseqlen-tlength; ++i) {
    if(mtuplesAll[0][i] >= 0) {
      int indices[100]; for(int j = 0; j < tupIdxcnt; ++j) { indices[j] = mtuplesAll[j+1][i]; }
      midx[mptr] = i;
      mptr++;
    }
  }

  for(int j = 0; j <= tupIdxcnt; ++j) {
    delete[] htuplesAll[j];
    delete[] mtuplesAll[j];
  }
  delete[] htuplesAll;
  delete[] mtuplesAll;
  delete ltt;
}

double Glass::findmap(int *seq1, int *seq2, 
		      int from1, int to1, int from2, int to2,  
		      double match, double mismatch, double gap, 
		      int *imagefirst, int *imagesecond, 
		      double *col1, double *col2, double *prebest, double *suffbest)
{  
  register int i,j, k;
  int lengthfirst  = to1 - from1 + 1;
  int lengthsecond = to2 - from2 + 1;
  if((lengthfirst == 0) || (lengthsecond == 0)) {
    return 0;
  }

  int m = (int) ceil(((double) lengthfirst) / 2) - 1 + from1;
    
  int mu = m - from1 + 1;
  
  double scorek[2]; 
  int maxindex    = 0; 
  int gapflag     = 0; 
  int tempgapflag = 0; 

  double maxscore=-Infinity;
  double up;
  double left;
  double upleft;
  
  k = lengthsecond;
  // Prefix Calculation
  // Boundary Conditions:
  prebest[0] = gap * (m - from1);
  
  for(i = 0; i <= mu - 1; ++i) {
    col1[i] = gap * i;
  }
  for(j = 1; j <= k; ++j) {
    col2[0] = gap * j;
    for(i = 1; i <= mu - 1; ++i) {
      up = col2[i-1] + gap;
      left = col1[i] + gap;
      upleft = col1[i-1] + ((seq1[from1 + i - 1] == seq2[from2 + j - 1]) ? match : mismatch);
      col2[i] = MAX(up, left, upleft);
      col1[i-1] = col2[i-1];
      if((i == mu-1) && (j != k)) {
	col1[mu-1] = col2[mu-1];
      }
    }
    prebest[j] = col2[mu-1]; 
  }
  if(m == from1) {
    for(j = 1; j <= k; ++j) {
      prebest[j] = gap * j;
    }
  }
  
  k=0;
  // Suffix Calculation
  // Boundary Conditions:
  suffbest[0] = gap * (to1 - m);
  for(i = 0; i <= to1 - m; ++i) {
    col1[i] = gap * i;
  }
  for(j = 1; j <= lengthsecond - k; ++j) {
    col2[0] = gap * j;
    for(i = 1; i <= to1 - m; ++i) {
      up = col2[i-1] + gap;
      left = col1[i] + gap;
      upleft = col1[i-1] + ((seq1[to1 - i + 1] == seq2[to2 - j + 1]) ? match : mismatch);
      col2[i] = MAX(up, left, upleft);
      col1[i-1] = col2[i-1];
      if((i == to1 - m) && (j != lengthsecond - k)) {
	col1[to1-m] = col2[to1-m];
      }
    }
    suffbest[j] = col2[to1-m];
  }
  if(m == to1) {
    for(j = 1; j <= lengthsecond; ++j) {
      suffbest[j] = gap * j;
    }
  }

  for(k = 0; k <= lengthsecond; ++k) {
    if(k > 0) {
      scorek[0] = prebest[k-1] + suffbest[lengthsecond-k] + 
	          ((seq1[m] == seq2[k+from2-1]) ? match : mismatch);
      scorek[1] = prebest[k] + suffbest[lengthsecond-k] + gap;
    }
    else if(k == 0) {
      scorek[0] =- Infinity;
      scorek[1] = prebest[0] + suffbest[lengthsecond] + gap;
    }
    
    if(scorek[0] >= scorek[1]) {
      tempgapflag=0;
    }
    else {
      tempgapflag=1;
    }
    if(scorek[tempgapflag] > maxscore) {
      maxscore = scorek[tempgapflag];
      maxindex = k + from2 - 1;
      gapflag  = tempgapflag;
    }
  }
  if(gapflag == 0) {
    imagefirst[m] = maxindex;
    imagesecond[imagefirst[m]] = m;
  }
  else {
    imagefirst[m] = -1;
  }

  if((m - 1 >= from1) && (from2 <= maxindex - 1 + gapflag)) {
    findmap(seq1, seq2, 
	    from1, m-1, from2, maxindex-1+gapflag, 
	    match, mismatch, gap, 
	    imagefirst, imagesecond, 
	    col1, col2, prebest, suffbest);
  }
  if((m + 1 <= to1) && (maxindex + 1 <= to2)) {
    findmap(seq1, seq2, 
	    m+1, to1, maxindex+1, to2,  
	    match, mismatch, gap, 
	    imagefirst, imagesecond, 
	    col1, col2, prebest, suffbest);
  }
  return maxscore;
}

double Glass::findmapUseExtensions(int *seq1, int *seq2, 
				   int from1, int to1, 
				   int from2, int to2,  
				   double match, double mismatch, double gap, 
				   int *imagefirst, int *imagesecond, 
				   double *col1, double *col2, double *prebest, double *suffbest,
				   int *hseqInt, int *mseqInt, 
				   int *hptrs, int *mptrs, 
				   int hseql, int mseql, 
				   int extnLength, int tupLength) 
{  
  register int i;
  register int j;
  register int k;
  int lengthfirst = to1 - from1 + 1;
  int lengthsecond = to2 - from2 + 1;
  if((lengthfirst == 0) || (lengthsecond == 0)) {
    return 0.0;
  }

  int m=(int)ceil(((double)lengthfirst)/2)-1+from1;
  int mu=m-from1+1;
  
  double scorek[2]; 
  int maxindex    = 0; 
  int gapflag     = 0; 
  int tempgapflag = 0; 
  
  double maxscore = -Infinity;
  double up, left, upleft;
  
  k = lengthsecond;
  // Prefix Calculation
  // Boundary Conditions:
  prebest[0]=gap*(m-from1);
  
  for(i = 0; i <= mu-1; ++i) {
    col1[i] = gap * i;
  }
  for(j = 1; j <= k; ++j) {
    col2[0] = gap * j;
    for(i = 1; i <= mu-1; ++i) {
      up   = col2[i-1] + gap;
      left = col1[i]   + gap;
      if(seq1[from1+i-1] == seq2[from2+j-1]) {
	double alignScoreLeft;
	double alignScoreRight;
	int* himageLeft = new int[extnLength]; int* himageRight = new int[extnLength];
	int* mimageLeft = new int[extnLength]; int* mimageRight = new int[extnLength];
	arrayZero(himageLeft,  extnLength); arrayZero(himageRight, extnLength);
	arrayZero(mimageLeft,  extnLength); arrayZero(mimageRight, extnLength);

	if((hptrs[from1+i-1] > extnLength) && (mptrs[from2+j-1] > extnLength)) {
	  alignScoreLeft = gappedAlign(hseqInt, mseqInt,
				       hptrs[from1+i-1] - extnLength, hptrs[from1+i-1] - 1,
				       mptrs[from2+j-1] - extnLength, mptrs[from2+j-1] - 1,
				       1, -1, -2, -1,
				       himageLeft, mimageLeft);
	}
	else {
	  alignScoreLeft = 0.0;
	}

	if ( (hptrs[from1+i-1] + tupLength + extnLength < hseql) && 
	     (mptrs[from2+j-1] + tupLength + extnLength < mseql) ) {
	  alignScoreRight = gappedAlign(hseqInt, mseqInt,
					hptrs[from1+i-1] + tupLength, 
					hptrs[from1+i-1] + tupLength + extnLength - 1,
					mptrs[from2+j-1] + tupLength, 
					mptrs[from2+j-1] + tupLength + extnLength - 1,
					1, -1, -2, -1,
					himageRight, mimageRight);
	}
	else {
	  alignScoreRight = 0;
	}
	
	delete[] himageLeft; delete[] himageRight;
	delete[] mimageLeft; delete[] mimageRight;
	upleft = col1[i-1] + alignScoreLeft + alignScoreRight;
      }
      else {
	upleft = col1[i-1] + mismatch;
      }
      col2[i]   = MAX(up, left, upleft);
      col1[i-1] = col2[i-1];
      if((i == mu-1) && (j != k)) {
	col1[mu-1] = col2[mu-1];
      }
    }
    prebest[j] = col2[mu-1]; 
  }
  if(m == from1) {
    for(j = 1; j <= k; ++j) {
      prebest[j] = gap * j;
    }
  }
  
  k=0;
  // Suffix Calculation
  // Boundary Conditions:
  suffbest[0] = gap * (to1-m);
  for(i = 0; i <= to1-m; ++i) {
    col1[i]=gap*i;
  }
  for(j = 1; j <= lengthsecond - k; ++j) {
    col2[0] = gap * j;
    for(i = 1; i <= to1 - m; ++i) {
      up   =col2[i-1] + gap;
      left =col1[i]   + gap;

      if(seq1[to1-i+1] == seq2[to2-j+1]) {
	double alignScoreLeft;
	double alignScoreRight;
	int* himageLeft = new int[extnLength]; int* himageRight = new int[extnLength];
	int* mimageLeft = new int[extnLength]; int* mimageRight = new int[extnLength];
	arrayZero(himageLeft,  extnLength); arrayZero(himageRight, extnLength);
	arrayZero(mimageLeft,  extnLength); arrayZero(mimageRight, extnLength);

	if((hptrs[to1-i+1] > extnLength) && (mptrs[to2-j+1] > extnLength)) {
	  alignScoreLeft = gappedAlign(hseqInt, mseqInt,
				       hptrs[to1-i+1] - extnLength, hptrs[to1-i+1] - 1,
				       mptrs[to2-j+1] - extnLength, mptrs[to2-j+1] - 1,
				       1, -1, -2, -1,
				       himageLeft, mimageLeft);
	}
	else {
	  alignScoreLeft = 0.0;
	}

	if((hptrs[to1+i-1] + tupLength + extnLength < hseql) && 
	   (mptrs[to2+j-1] + tupLength + extnLength < mseql)) {
	  alignScoreRight = gappedAlign(hseqInt, mseqInt,
					hptrs[to1+i-1] + tupLength, hptrs[to1+i-1] + tupLength + extnLength - 1,
					mptrs[to2+j-1] + tupLength, mptrs[to2+j-1] + tupLength + extnLength - 1,
					1, -1, -2, -1,
					himageRight, mimageRight);
	}
	else {
	  alignScoreRight = 0.0;
	}
	delete[] himageLeft; delete[] himageRight;
	delete[] mimageLeft; delete[] mimageRight;
	upleft = col1[i-1] + alignScoreLeft + alignScoreRight;
      }
      else {
	upleft = col1[i-1] + mismatch;
      }
      col2[i]   = MAX(up,left,upleft);
      col1[i-1] = col2[i-1];
      if((i == to1 - m) && (j != lengthsecond - k)) {
	col1[to1-m]=col2[to1-m];
      }
    }
    suffbest[j] = col2[to1-m];
  }
  if(m == to1) {
    for(j = 1; j <= lengthsecond; ++j) {
      suffbest[j] = gap * j;
    }
  }
  for(k = 0; k <= lengthsecond; ++k) {
    if(k > 0) {
      scorek[0] = prebest[k-1] + 
	          suffbest[lengthsecond-k] +
	          ((seq1[m]==seq2[k+from2-1]) ? match : mismatch);
      scorek[1] = prebest[k] +
	          suffbest[lengthsecond-k] + 
	          gap;
    }
    else if(k == 0) {
      scorek[0] = -Infinity;
      scorek[1] = prebest[0] + suffbest[lengthsecond] + gap;
    }
    
    if(scorek[0] >= scorek[1]) {
      tempgapflag = 0;
    }
    else {
      tempgapflag = 1;
    }
    if(scorek[tempgapflag] > maxscore) {
      maxscore = scorek[tempgapflag];
      maxindex = k + from2 - 1;
      gapflag  = tempgapflag;
    }
  }
  if(gapflag == 0) {
    imagefirst[m] = maxindex;
    imagesecond[imagefirst[m]]=m;
  }
  else {
    imagefirst[m] = -1;
  }
  
  if((m-1 >= from1) && (from2 <= maxindex-1+gapflag)) {
    findmapUseExtensions(seq1, seq2, from1, m-1, from2, maxindex-1+gapflag, match, mismatch, gap,
			 imagefirst, imagesecond, col1, col2, prebest, suffbest,
			 hseqInt, mseqInt, hptrs, mptrs, hseql, mseql, extnLength, tupLength);
  }
  if((m+1 <= to1) && (maxindex+1 <= to2)) {
    findmapUseExtensions(seq1, seq2, m+1, to1, maxindex+1, to2,  match, mismatch, gap,
			 imagefirst, imagesecond, col1, col2, prebest, suffbest,
			 hseqInt, mseqInt, hptrs, mptrs, hseql, mseql, extnLength, tupLength);
  }

  return maxscore;
}


double Glass::memalign(int *first, int *second, 
		       int fromfirst, int tofirst, 
		       int fromsecond, int tosecond, 
		       double match, double mismatch, double gap, 
		       int *imagefirst, int *imagesecond)
{
  double score;
  int lengthfirst  = tofirst  - fromfirst  + 1;
  int lengthsecond = tosecond - fromsecond + 1;
  double* col1 = new double[lengthfirst+1]; arrayZero(col1, lengthfirst+1);
  double* col2 = new double[lengthfirst+1]; arrayZero(col2, lengthfirst+1);
  double* prebest  = new double[lengthsecond+1]; arrayZero(prebest,  lengthsecond+1);
  double* suffbest = new double[lengthsecond+1]; arrayZero(suffbest, lengthsecond+1);
  
  for(int i = 0; i < lengthfirst; ++i) {
    imagefirst[i] = -1;
  }
  for(int i = 0; i < lengthsecond; ++i) {
    imagesecond[i]= -1;
  }

  score = findmap(first+fromfirst, second+fromsecond, 
		  0, tofirst-fromfirst, 
		  0, tosecond-fromsecond,
		  match, mismatch, gap, 
		  imagefirst, imagesecond, 
		  col1, col2, prebest, suffbest);
  
  delete[] col1;
  delete[] col2;
  delete[] prebest;
  delete[] suffbest;
  
  return score;
}

double Glass::gappedAlign(int *first, int *second, 
			  int fromfirst, int tofirst, int fromsecond, int tosecond,
			  double match, double mismatch, double gapOpen, double gap, 
			  int *imagefirst, int *imagesecond) 
{  

  if ( (tofirst - fromfirst) * (tosecond - fromsecond) > 2048 * 2048 ) {
    warn("alignment too big for gappedAlign. Using memalign instead !");
    return memalign(first, second, 
		    fromfirst, tofirst, fromsecond, tosecond, 
		    match, mismatch, gap, imagefirst, imagesecond);
  }
 
  int lengthfirst  = tofirst  - fromfirst  + 1;
  int lengthsecond = tosecond - fromsecond + 1;

  double score = 0.0;

  int newlen1 = 100;
  if(lengthfirst+2 > newlen1) { newlen1 = lengthfirst+2; }

  int newlen2 = 100;
  if(lengthsecond+2 > newlen2) { newlen2 = lengthsecond+2; }

  if(MatrixM == NULL)
    MatrixM = new (double**)[2];
  if(MatrixP == NULL)
    MatrixP = new (int**)[2];

  // if either grows reallow whole mess, min of 100
  if((newlen1 > len1) || (newlen2 > len2)) {
    double** temp1 = new (double*)[newlen1];
    double** temp2 = new (double*)[newlen1];
    int** itemp1 = new (int*)[newlen1];
    int** itemp2 = new (int*)[newlen1];

    // make new l2 arrays
    for(int i = 0; i < newlen1;i++){
      temp1[i] = new (double)[newlen2];
      temp2[i] = new (double)[newlen2];
      itemp1[i] = new (int)[newlen2];
      itemp2[i] = new (int)[newlen2];
    }

    for(int i = 0; i < len1; i++) {
      delete[] MatrixM[0][i]; MatrixM[0][i] = 0;
      delete[] MatrixM[1][i]; MatrixM[1][i] = 0;
      delete[] MatrixP[0][i]; MatrixP[0][i] = 0;
      delete[] MatrixP[1][i]; MatrixP[1][i] = 0;
    } 
    if(len1 > 0){
      delete[] MatrixM[0]; 
      delete[] MatrixM[1]; 
      delete[] MatrixP[0]; 
      delete[] MatrixP[1]; 
    }
  
    MatrixM[0] = temp1;
    MatrixM[1] = temp2;
    MatrixP[0] = itemp1;
    MatrixP[1] = itemp2;
    len1 = newlen1;
    len2 = newlen2;
  }

  for(int i = 0; i < lengthfirst + 2; ++i) {
    arrayZero(MatrixM[0][i], lengthsecond+2);
    arrayZero(MatrixM[1][i], lengthsecond+2);
  }
  for(int i = 0; i < lengthfirst + 2; ++i) {
    arrayZero(MatrixP[0][i], lengthsecond+2);
    arrayZero(MatrixP[1][i], lengthsecond+2);
  }

  // Boundary Conditions:
  MatrixM[0][0][0] = 0;
  MatrixM[1][0][0] = -Infinity;
  MatrixP[0][0][0] = MatrixP[1][0][0] = 0;

  for(int i = 1; i < lengthfirst + 1; ++i) {
    MatrixM[0][i][0] = -Infinity; MatrixM[1][i][0] = gap*i + gapOpen-gap;
    MatrixP[0][i][0] = toU1;      MatrixP[1][i][0] = toU1;
  }
  MatrixP[1][0][0] = toU0;

  for(int i = 1; i < lengthsecond + 1; ++i) {
    MatrixM[0][0][i] = -Infinity; MatrixM[1][0][i] = gap*i + gapOpen-gap;
    MatrixP[0][0][i] = toL1;      MatrixP[1][0][i] = toL1;
  }
  MatrixP[0][0][1] = toL0;

  for(int i = 1; i < lengthfirst + 1; ++i) {
    for(int j = 1; j < lengthsecond + 1; ++j) {
      
      int i_1 = i-1;
      int j_1 = j-1;
      
      double U0 = MatrixM[ 0 ][i_1][ j ],   U1 = MatrixM[ 1 ][i_1][ j ];
      double L0 = MatrixM[ 0 ][ i ][j_1],   L1 = MatrixM[ 1 ][ i ][j_1];
      double D0 = MatrixM[ 0 ][i_1][j_1],   D1 = MatrixM[ 1 ][i_1][j_1];
      
      double d = (first[fromfirst+i_1]==second[fromsecond+j_1]) ? match : mismatch;
      double g = gapOpen;
      double e = gap;
      
      MatrixM[0][i][j] = MAX( D0 , D1 ) + d;

      if (D0>=D1) {
	MatrixP[0][i][j] = toD0;
      }
      else {
	MatrixP[0][i][j] = toD1;
      }

      MatrixM[1][i][j] = MAX( MAX( U0,L0 ) + g   ,   MAX( U1,L1 ) + e );

      if      (U0 >= L0 && U0+g >= MAX(U1,L1)+e)   { MatrixP[1][i][j] = toU0; }
      else if (U1 >= L1 && U1+e >=     L0    +g)   { MatrixP[1][i][j] = toU1; }
      else if (            L0+g >=     L1    +e)   { MatrixP[1][i][j] = toL0; }
      else                                         { MatrixP[1][i][j] = toL1; }
    }
  }
   // find optimal alignment and fill in the arrays imagefirst and imagesecond
  int i = lengthfirst;
  int j = lengthsecond;
  int l = (MatrixM[0][i][j] >= MatrixM[1][i][j]) ? 0 : 1;
  
  score = MatrixM[l][i][j];

  while (i > 0 || j > 0) {

    switch( MatrixP[l][i][j] ) {

    case toU0:
      imagefirst[--i] = -1;
      l = 0;
      break;

    case toU1:
      imagefirst[--i] = -1;
      l = 1;
      break;
    
    case toL0:
      imagesecond[--j] = -1;
      l = 0;
      break;

    case toL1:
      imagesecond[--j] = -1;
      l = 1;
      break;

    case toD0:
      imagefirst[i-1]  = j-1;
      imagesecond[--j] = --i;
      l = 0;
      break;
    
    case toD1:
      imagefirst[i-1]  = j-1;
      imagesecond[--j] = --i;
      l = 1;
      break;
      
    default: assert(0);
    }
  }
  return score;
}


void Glass::constraintsalignUseExtensions(int *first, int *second, 
					  int length1, int length2, 
					  double match, double mismatch, double gap, 
					  int *imagefirst, int *imagesecond, 
					  int *constraints1, int *constraints2, 
					  int constraintslength, int acceptNegativeAlignments,
					  int *hseqInt, int *mseqInt, 
					  int *hptrs, int *mptrs,
					  int hseql, int mseql, 
					  int extnLength, int tupLength) 
{
  register int i;
  for(i = 0; i < length1; ++i) {
    imagefirst[i] = -1;
  }
  for(i = 0; i < length2; ++i) {
    imagesecond[i]=-1;
  }
  
  int from1 = 0;
  int from2 = 0;
  int to1   = 0;
  int to2   = 0;
  double piecescore = -1; 
  double* col1     = new double[length1+1]; arrayZero(col1, length1+1);
  double* col2     = new double[length1+1]; arrayZero(col2, length1+1);
  double* prebest  = new double[length2+1]; arrayZero(prebest, length2+1);
  double* suffbest = new double[length2+1]; arrayZero(suffbest, length2+1);
  
  // begin alignment
  for(i = 0; i < constraintslength; ++i){
    to1 = constraints1[i] - 1;
    to2 = constraints2[i] - 1;
    imagefirst[constraints1[i]]  = constraints2[i];
    imagesecond[constraints2[i]] = constraints1[i];
    if(!((from1==constraints1[i]) || (from2==constraints2[i]))) {
      if((i == 0) && ((from1 > to1) || (from2 > to2))) { continue; }
      
      piecescore = findmapUseExtensions(first, second, from1, to1, from2, to2,
					match, mismatch, gap, imagefirst, imagesecond, 
					col1, col2, prebest, suffbest,
					hseqInt, mseqInt, hptrs, mptrs,
					hseql, mseql, extnLength, tupLength);
    }
    from1 = constraints1[i]+1;
    from2 = constraints2[i]+1;
  }
  if((from1 <= length1 - 1) && (from2 <= length2 - 1)) {
    to1 = length1 - 1;
    to2 = length2 - 1;
    findmapUseExtensions(first, second, from1, to1, from2, to2, match, mismatch, gap,
			 imagefirst, imagesecond, col1, col2, prebest, suffbest,
			 hseqInt, mseqInt, hptrs, mptrs, hseql, mseql, extnLength, tupLength);
  }
  delete[] col1;
  delete[] col2;
  delete[] prebest;
  delete[] suffbest;
}


double Glass::tupleAlignmentScore(int *htup, int *mtup, 
				  int hfrom, int hto, int mfrom, int mto,
				  double match, double mismatch, double gap, double alpha, double beta,
				  int *himg, int *mimg) 
{
  double score = 0.0;

  int matchCount = 0;

  for(int i = hfrom; i <= hto; ++i) {
    if(himg[i] >= 0) {
      if(htup[i] == mtup[himg[i]]) {
	score += match * power(beta, matchCount);
	matchCount++;
      }
      else {
	score += mismatch;
	matchCount = 0;
      }
    }
    if(score >= REALLYBIG) {
      break;  // Quick hack to avoid overflow on perfect matches.
    }
  }

  int gapCount = 0;
  for(int i = hfrom; i <= hto; ++i) {
    if(himg[i] >= 0) {
      gapCount=0;
    }
    else {
      score += gap * power(alpha, gapCount);
      gapCount++;
    }
    if(score >= REALLYBIG) {
      break;    // Quick hack to avoid overflow on perfect matches.
    }
  }

  gapCount = 0;
  for(int i = mfrom; i <= mto; ++i) {
    if(mimg[i] >= 0) {
      gapCount=0;
    }
    else {
      score += gap * power(alpha, gapCount);
      gapCount++;
    }
    if(score >= REALLYBIG) {
      break;    // Quick hack to avoid overflow on perfect matches.
    }
  }

  return score;
}

//
// I don't know where this is supposed to come from
int max(int a, int b){
  if(a>b) return(a);
  return(b);
}









